const WebSocket = require('ws');
const minesweeper = require('minesweeper');
const DEBUG = false;

const wss = new WebSocket.Server({ port: 8081 });
const matches = [];

console.log("Server online");

wss.on('connection', function connection(ws,httpRequest) {
  let urlSplit = httpRequest.url.split("?");
  let reqURL = new URLSearchParams(urlSplit[1]);

  ws.clientIPAddr = httpRequest.connection.remoteAddress;

  if (ws.clientIPAddr == "::ffff:73.241.108.46"){
    ws.isEast = true;
    ws.send(JSON.stringify({arbitraryText: "Hey East, you're cute ❤️\nDon't worry nobody else sees this :p"}));
  }

  if (urlSplit[0] == "/host"){
    setupNewMatch(ws,reqURL);
  }

  ws.on('message', function incoming(message) {
    let gameMessage = JSON.parse(message);
    if (gameMessage.type !== "mousemove"){
      console.log('Incoming:', message);
    }
    if (gameMessage.nickname){
      console.log(gameMessage.nickname,ws.clientIPAddr);
    }

    let matchID = gameMessage.ID;

    if (matches[matchID] == undefined){
      ws.send(JSON.stringify({error: "Invalid Game ID"}));
      return;
    }

    let match = matches[matchID];
    let board = match.game;
    let error = null;
    let gameOver = board && (board.state() === minesweeper.BoardStateEnum.WON || board.state() === minesweeper.BoardStateEnum.LOST);

    if (gameMessage.type === 'newgame'){
      configureNewGame(matchID,gameMessage.rows,gameMessage.columns,gameMessage.mines,ws.isEast);
      broadcastGameState(gameMessage,match,ws.playerNick,error);
    }
    else if (gameMessage.type === 'joingame'){
      return handleJoinGame(ws,matchID,match,gameMessage.nickname);
    }
    else if (gameMessage.type === 'flag' && !gameOver){
      board.cycleCellFlag(gameMessage.X,gameMessage.Y);
      broadcastGameState(gameMessage,match,ws.playerNick,error);
    }
    else if (gameMessage.type === 'open' && !gameOver){
      board.openCell(gameMessage.X,gameMessage.Y);
      broadcastGameState(gameMessage,match,ws.playerNick,error);
    }
    else if (gameMessage.type === 'mousemove'){
      return handleMouseMove(match,ws,gameMessage.mouseX,gameMessage.mouseY);
    }
  });
});

function broadcastGameState(gameData,match,moveOwner,error = null){
  let response = JSON.stringify({moveOwner: moveOwner,
    state: match.game.state(), grid: match.game.grid(),
    players: match.players.map(x => (x.playerNick)), error: error,
    origX: gameData.X, origY: gameData.Y
  });

  match.players.forEach( (player,index) => {
    player.send(response,function(err){
      if (err) cleanUpSocket(index,match.players);
    })
  });
}

function setupNewMatch(socket,queryparams){
  let nick = queryparams.get('nickname').trim().substring(0,16).replace(/[^\x00-\x7F]/g, "");;
  if (!nick || nick == "") return;
  socket.playerNick = nick;
  let matchID = Math.floor(Math.random() * 1000);
  matches[matchID] = {players: [socket]};
  socket.send(JSON.stringify({ID: matchID, players:[nick]}));
}

function configureNewGame(matchID,rows,columns,mines,isEast = false){
  let mineArray = [];
  if (true){
    mineArray = minesweeper.generateMineArray(
      {
        rows: Math.abs(rows) % 50,
        cols: Math.abs(columns) % 50,
        mines: Math.min(Math.abs(mines), rows * columns)
      }
    );
  }
  else{
    mineArray.length = 210;
    mineArray.fill(0);
    mineArray[17] = 1;
    mineArray[20] = 1;
    mineArray[24] = 1;
    mineArray[25] = 1;
    mineArray[26] = 1;
    mineArray[32] = 1;
    mineArray[35] = 1;
    mineArray[39] = 1;
    mineArray[42] = 1;
    mineArray[47] = 1;
    mineArray[50] = 1;
    mineArray[54] = 1;
    mineArray[55] = 1;
    mineArray[56] = 1;
    mineArray[62] = 1;
    mineArray[65] = 1;
    mineArray[69] = 1;
    mineArray[70] = 1;
    mineArray[77] = 1;
    mineArray[80] = 1;
    mineArray[84] = 1;
    mineArray[86] = 1;
    mineArray[92] = 1;
    mineArray[93] = 1;
    mineArray[94] = 1;
    mineArray[95] = 1;
    mineArray[99] = 1;
    mineArray[102] = 1;

    mineArray[120] = 1;
    mineArray[121] = 1;
    mineArray[122] = 1;
    mineArray[124] = 1;
    mineArray[126] = 1;
    mineArray[128] = 1;
    mineArray[129] = 1;
    mineArray[130] = 1;
    mineArray[132] = 1;
    mineArray[133] = 1;
    mineArray[134] = 1;
    mineArray[135] = 1;
    mineArray[139] = 1;
    mineArray[141] = 1;
    mineArray[144] = 1;
    mineArray[147] = 1;
    mineArray[150] = 1;
    mineArray[154] = 1;
    mineArray[156] = 1;
    mineArray[159] = 1;
    mineArray[162] = 1;
    mineArray[163] = 1;
    mineArray[164] = 1;
    mineArray[165] = 1;
    mineArray[169] = 1;
    mineArray[171] = 1;
    mineArray[174] = 1;
    mineArray[177] = 1;
    mineArray[180] = 1;
    mineArray[181] = 1;
    mineArray[182] = 1;
    mineArray[184] = 1;
    mineArray[185] = 1;
    mineArray[186] = 1;
    mineArray[189] = 1;
    mineArray[192] = 1;
    mineArray[193] = 1;
    mineArray[194] = 1;

    mineArray = singleToMultiDimensionalArray(mineArray, 15);
  }
  let game = new minesweeper.Board(mineArray);
  matches[matchID].game = game;
}

function cleanUpSocket(playerIndex,playerArray){
  setTimeout(() => {
    delete playerArray[playerIndex]
  },50);
}

function handleJoinGame(ws, matchID, match, nickname){
  let nick = nickname.trim().substring(0,16).replace(/[^\x00-\x7F]/g, "");
  if (!nick || nick == "" || nickInUse(nick,match)) return;
  ws.playerNick = nick;
  match.players.push(ws);
  let response = JSON.stringify({ID: matchID, state: match.game.state(),
    grid: match.game.grid(), players: match.players.map(x => (x.playerNick))
  });
  ws.send(response);
}

function handleMouseMove(match,ws,mouseX,mouseY){
  let response = JSON.stringify({playerMouseX: mouseX, playerMouseY: mouseY, player: ws.playerNick});
  match.players.forEach((player) => {
    if (player == ws) return;
    player.send(response);
  });
}

function nickInUse(nick,match){
  match.players.forEach(player => {
    if (player.playerNick == nick) return true;
  })
  return false;
}

function singleToMultiDimensionalArray(array, numCols) {
  var i,
      rows = array.length / numCols,
      multi = [];

  for (i = 0; i < rows; i++) {
    multi.push(array.splice(0, numCols));
  }

  return multi;
};
