const CellStateEnum = {CLOSED: 0,OPEN: 1};
const CellFlagEnum = {NONE: 0,EXCLAMATION: 1,QUESTION: 2};
const BoardStateEnum = {PRISTINE: 0,IN_PROGRESS: 1,LOST: 2,WON: 3};

let ws;
let gameID;
let players = {};

$(document).ready(function(){
  $('#joingame').click(joinGame);
  $('#creategame').click(createGame);
  $('#newgame').click(newGame);
  $('#noofrows, #noofcols').focusout(calcSuggestedMines);
  $('#zoomfactor').on('input', alterZoom);
  let parsedURL = new URL(document.location);
  let joinCode = parsedURL.searchParams.get('join');
  if (!isNaN(parseInt(joinCode))){
    $('#joincode').val(joinCode);
  }
});

function generateMinesweeperGrid(rows,columns){
  let $mineSweeper = $('#minesweeper').eq(0);
  $mineSweeper.empty();
  $mineSweeper.off('mousemove');
  for (let row = 0; row < rows; row++) {
    let rowHTML = $('<tr></tr>').appendTo($mineSweeper);
    for (let column = 0; column < columns; column++) {
      let cellHTML = $('<td><p class="cellContent"></p></td>').addClass('cell').data('XPos',column).data('YPos',row);
      rowHTML.append(cellHTML);
    }
  }
  $mineSweeper.show();
  $('.cell').click(cellClick).contextmenu(cellClick);
  $mineSweeper.mousemove(sendMouseCursor);
}

function handleGameData(data){
  let message = JSON.parse(data);
  console.log("Incoming: ", message);
  if (message.arbitraryText){
    return alert(message.arbitraryText);
  }
  gameID = message.ID || gameID;
  $('#gameid').text(gameID);
  if (message.playerMouseX != undefined){
    renderMouseCursors(message);
  }
  else{
    updatePlayers(message.players,message.moveOwner);
    parseMessage(message.grid,message.state);
    setCellColour(message.moveOwner,message.origX,message.origY,message.grid && message.grid[0].length);
  }
}

function cellClick(event){
  let $cell = $(this);
  let XPos = $cell.data('XPos');
  let YPos = $cell.data('YPos');
  let moveData = {ID: gameID, X: XPos, Y: YPos};
  moveData.type = event.type === "click" ? 'open' : 'flag';
  console.log(moveData);
  ws.send(JSON.stringify(moveData));
  return false;
}

function parseMessage(grid,state){
  if (state === BoardStateEnum.WON || state === BoardStateEnum.LOST){
    renderBoard(grid,true);
    timerControl(false);
  }
  else if (state === BoardStateEnum.PRISTINE){
    $('#zoomfactor').val(1)
    generateMinesweeperGrid(grid.length,grid[0].length);
    timerControl(true);
  }
  else if (state === BoardStateEnum.IN_PROGRESS){
    renderBoard(grid);
  }
}

function renderBoard(grid,gameover=false){
  let $cells = $('.cell');
  for (let row = 0; row < grid.length; row++) {
    for (let column = 0; column < grid[row].length; column++) {
      const cell = grid[row][column];
      const htmlCell = $cells.eq(row * grid[0].length + column);
      formatCell(htmlCell,cell,gameover);
    }
  }
}

function formatCell(htmlCell,cell,gameover){
  if (cell.state == CellStateEnum.OPEN){
    htmlCell.addClass('open');
    if (cell.isMine){
      htmlCell.addClass('exploded');
    }
    $('.cellContent',htmlCell).text(cell.numAdjacentMines > 0 ? cell.numAdjacentMines : "");
    $('.cellContent',htmlCell).addClass('mine' + cell.numAdjacentMines);
  }
  else{
    switch (cell.flag) {
      case CellFlagEnum.EXCLAMATION:
        $('.cellContent',htmlCell).text("!");
        htmlCell.addClass('flagged');
        break;
      case CellFlagEnum.QUESTION:
        $('.cellContent',htmlCell).text("?");
        htmlCell.addClass('flagged');
        break;
      case CellFlagEnum.NONE:
        $('.cellContent',htmlCell).text("");
        htmlCell.removeClass('flagged');
    }
    if (gameover && cell.isMine){
      htmlCell.addClass('mine');
    }
  }
}

function setCellColour(moveOwner,column,row,rowcount){
  let playerID = players[moveOwner];
  console.log(playerID,row * rowcount + column);
  if (!playerID || column == undefined) return;
  $('.cell > .cellContent').eq(row * rowcount + column).attr('data-player-colour', playerID);
}

function createGame(){
  let $nickname = $('#nickname');
  if (!$nickname[0].reportValidity()) return false;
  $('#joincontrols').hide();
  ws = new WebSocket(`ws://vps341636.ovh.net:8081/host?nickname=${$nickname.val()}`);
  ws.onopen = function () {
    console.log("CONNECTION ESTABLISHED", ws);
    $('#gamecontrols').show();
  };
  ws.onmessage = function (event) {
    handleGameData(event.data);
  };
  return false;
}

function newGame(){
  if (document.forms['gamecontrols'].reportValidity()){
    let columns = $('#noofcols').val();
    let rows = $('#noofrows').val();
    let mines = $('#noofmines').val();
    let message = {
      ID: gameID,
      type: 'newgame',
      rows: rows,
      columns: columns,
      mines: mines
    };
    ws.send(JSON.stringify(message));
  }
  return false;
}

function joinGame(){
  if (document.forms['joincontrols'].reportValidity()){
    $('#joincontrols').hide();
    ws = new WebSocket("ws://vps341636.ovh.net:8081/join");
    ws.onopen = function () {
      console.log("CONNECTION ESTABLISHED", ws);
      let message = {};
      message.ID = $('#joincode').val();
      message.type = 'joingame';
      message.nickname = $('#nickname').val();
      ws.send(JSON.stringify(message));
    };
    ws.onmessage = function (event) {
      let message = JSON.parse(event.data);
      console.log(message);
      if (message.playerMouseX) return;
      if (message.arbitraryText){
        return alert(message.arbitraryText);
      }
      gameID = message.ID;
      generateMinesweeperGrid(message.grid.length,message.grid[0].length);
      let gameOver = message.state == BoardStateEnum.WON || message.state == BoardStateEnum.LOST;
      renderBoard(message.grid,gameOver);
      $('#gameid').text(gameID);
      updatePlayers(message.players);
      ws.onmessage = function(event){
        handleGameData(event.data);
      }
    };
  }
  return false;
}

function calcSuggestedMines(){
  let columns = $('#noofcols').val();
  let rows = $('#noofrows').val();
  let suggestedMines = columns * rows / 6.4;
  if (!isNaN(parseInt(suggestedMines))){
    mines = $('#noofmines').val(Math.round(suggestedMines));
  }
}

function updatePlayers(playerArray,lastMovePlayer){
  $('#players').empty();
  let prevPlayerID = "";
  for (let index = 0; index < playerArray.length; index++) {
    const nick = playerArray[index];
    if (nick == null) continue;
    const colourID = (index + 1);
    if (nick == lastMovePlayer) prevPlayerID = colourID;
    $('<p></p>').addClass('playername').attr('data-player-colour',colourID).text(nick).appendTo('#players');
    players[nick] = index + 1;
  }
  $('#prevMove').text(lastMovePlayer || "").attr('data-player-colour',prevPlayerID);
}

function sendMouseCursor(event){
  let gridPos = $(this).offset(); // Get position of minesweeper grid on page
  let borderWidth = parseFloat($(this).css("border-top-width"),10); // Get width of border
  let message = { //construct message to send to server
    ID: gameID,
    type: 'mousemove',
    // Get mouse X and Y, subtract grid position to make relvative to grid & remove border
    // then divide by width of the grid to make proportional, then * 100 to get percentage
    mouseX: (event.pageX - gridPos.left - borderWidth) / ($(this).width()) * 100,
    mouseY: (event.pageY - gridPos.top - borderWidth) / ($(this).height()) * 100
  };
  ws.send(JSON.stringify(message));
}

function renderMouseCursors(message){
  let playerID = players[message.player];
  if (!playerID) return;
  let $existingCursor = $(`div[data-player-mouse=${playerID}]`);
  console.log(message.playerMouseX + '%',message.playerMouseY + '%')
  if ($existingCursor.length == 1){
    $existingCursor.css('left',message.playerMouseX + '%').css('top',message.playerMouseY + '%');
  }
  else{
    $('<div></div>').addClass('mouseCursor').css('left',message.playerMouseX + '%')
    .css('top',message.playerMouseY + '%').attr('data-player-mouse',playerID)
    .css('background-image',`url(cursors/${playerID}.png`).appendTo('#minesweeper');
  }
}

let gameTimer;
function timerControl(start){
  if (start){
    clearTimeout(gameTimer);
    gameTimer = setTimeout(updateTimer,100,0);
  }
  else{
    clearTimeout(gameTimer);
  }
}

function updateTimer(currentTime){
  currentTime += 0.1;
  $('#gametimer').text(currentTime.toFixed(1) + 's');
  gameTimer = setTimeout(updateTimer,100,currentTime);
}

function alterZoom(){
  let $cells = $('.cell');
  let $cellContents = $('.cellContent');
  let zoomFactor = +$(this).val();
  let widthCalc = (2 * zoomFactor) + 'vw';
  let heightCalc = (2.4 * zoomFactor) + 'vw';
  let borderWidth = (0.2 * zoomFactor) + 'vw';
  let fontSize = (1.5 * zoomFactor) + 'vw';
  console.log(zoomFactor,widthCalc,heightCalc,borderWidth,fontSize);
  $cells.width(widthCalc).height(heightCalc).css('border-width',borderWidth);
  $cellContents.width(widthCalc).height(widthCalc).css('font-size',fontSize).css('line-height',widthCalc);
}